-- PyrosparkCore - Client Commands
-- Main command: /ps [subcommand]

-- Helper function for notifications
local function ShowNotification(message)
    if Config.ShowNotifications then
        SetNotificationTextEntry('STRING')
        AddTextComponentString(message)
        DrawNotification(false, false)
    end
end

-- Helper function to send chat message
local function SendChatMessage(message, color)
    color = color or {255, 255, 255}
    TriggerEvent('chat:addMessage', {
        color = color,
        args = {'^3[PyrosparkCore]^7', message}
    })
end

-- ============================================================================
--                          MAIN /PS COMMAND
-- ============================================================================

RegisterCommand('ps', function(source, args, rawCommand)
    local subcommand = args[1]

    if not subcommand then
        -- No subcommand provided, show help
        TriggerEvent('ps:help')
        return
    end

    -- Convert to lowercase for case-insensitive matching
    subcommand = string.lower(subcommand)

    -- Route to appropriate subcommand
    if subcommand == 'help' then
        TriggerEvent('ps:help')
    elseif subcommand == 'dv' then
        TriggerEvent('ps:dv')
    elseif subcommand == 'seatdrive' then
        TriggerEvent('ps:seatdrive')
    elseif subcommand == 'fix' then
        TriggerEvent('ps:fix')
    elseif subcommand == 'refuel' then
        TriggerEvent('ps:refuel')
    elseif subcommand == 'coords' then
        TriggerEvent('ps:coords')
    elseif subcommand == 'whoami' then
        TriggerServerEvent('ps:whoami')
    else
        SendChatMessage('Unknown command. Use ^3/ps help^7 for available commands.', {255, 0, 0})
    end
end)

-- ============================================================================
--                          HELP COMMAND
-- ============================================================================

RegisterNetEvent('ps:help', function()
    SendChatMessage('=== Available Commands ===', {0, 255, 0})

    -- Check permissions and only show available commands
    TriggerServerEvent('ps:requestHelp')
end)

RegisterNetEvent('ps:showHelp', function(availableCommands)
    for _, cmd in ipairs(availableCommands) do
        SendChatMessage(cmd.command .. ' - ' .. cmd.description)
    end
    SendChatMessage('Use ^3/ps [command]^7 to run a command')
end)

-- ============================================================================
--                          VEHICLE COMMANDS
-- ============================================================================

-- ps dv - Delete Vehicle
RegisterNetEvent('ps:dv', function()
    local playerPed = PlayerPedId()
    local vehicle = nil

    -- Check if player is in a vehicle
    if IsPedInAnyVehicle(playerPed, false) then
        vehicle = GetVehiclePedIsIn(playerPed, false)
    else
        -- Get closest vehicle
        local coords = GetEntityCoords(playerPed)
        vehicle = GetClosestVehicle(coords.x, coords.y, coords.z, Config.DeleteVehicleRange, 0, 71)
    end

    if vehicle and vehicle ~= 0 and DoesEntityExist(vehicle) then
        -- Check if vehicle is empty (if config requires it)
        if Config.DeleteEmptyOnly then
            local driver = GetPedInVehicleSeat(vehicle, -1)
            if driver ~= 0 then
                ShowNotification('~r~Vehicle must be empty to delete')
                return
            end
        end

        -- Delete the vehicle
        SetEntityAsMissionEntity(vehicle, true, true)
        DeleteVehicle(vehicle)
        ShowNotification('~g~Vehicle deleted')

        if Config.DebugMode then
            print('[PyrosparkCore] Vehicle deleted by player')
        end
    else
        ShowNotification('~r~No vehicle found nearby')
    end
end)

-- ps seatdrive - Move to Driver Seat
RegisterNetEvent('ps:seatdrive', function()
    local playerPed = PlayerPedId()

    if IsPedInAnyVehicle(playerPed, false) then
        local vehicle = GetVehiclePedIsIn(playerPed, false)

        -- Check if driver seat is free
        local driver = GetPedInVehicleSeat(vehicle, -1)
        if driver == 0 or driver == playerPed then
            SetPedIntoVehicle(playerPed, vehicle, -1)
            ShowNotification('~g~Moved to driver seat')
        else
            ShowNotification('~r~Driver seat is occupied')
        end
    else
        ShowNotification('~r~You must be in a vehicle')
    end
end)

-- ps fix - Repair Vehicle
RegisterNetEvent('ps:fix', function()
    local playerPed = PlayerPedId()

    if IsPedInAnyVehicle(playerPed, false) then
        local vehicle = GetVehiclePedIsIn(playerPed, false)

        if Config.FixFullRepair then
            -- Full repair
            SetVehicleFixed(vehicle)
            SetVehicleDeformationFixed(vehicle)
            SetVehicleUndriveable(vehicle, false)
            SetVehicleEngineOn(vehicle, true, true)
        end

        if Config.FixCleansVehicle then
            SetVehicleDirtLevel(vehicle, 0.0)
            WashDecalsFromVehicle(vehicle, 1.0)
        end

        ShowNotification('~g~Vehicle repaired')

        if Config.DebugMode then
            print('[PyrosparkCore] Vehicle repaired')
        end
    else
        ShowNotification('~r~You must be in a vehicle')
    end
end)

-- ============================================================================
--                          UTILITY COMMANDS
-- ============================================================================

-- ps refuel - Refuel Vehicle
RegisterNetEvent('ps:refuel', function()
    local playerPed = PlayerPedId()

    if IsPedInAnyVehicle(playerPed, false) then
        local vehicle = GetVehiclePedIsIn(playerPed, false)

        -- Trigger VehicleHUD refuel event if it's running
        if GetResourceState('PS-VehicleHUD') == 'started' then
            TriggerEvent('vehiclehud:adminRefuel', vehicle)
            ShowNotification('~g~Vehicle refueled to 100%')
        else
            ShowNotification('~r~VehicleHUD is not running')
        end

        if Config.DebugMode then
            print('[PyrosparkCore] Vehicle refueled via command')
        end
    else
        ShowNotification('~r~You must be in a vehicle')
    end
end)

-- ps coords - Get Current Coordinates
RegisterNetEvent('ps:coords', function()
    local playerPed = PlayerPedId()
    local coords = GetEntityCoords(playerPed)
    local heading = GetEntityHeading(playerPed)

    print('=== PLAYER COORDINATES ===')
    print(string.format('vector3(%.2f, %.2f, %.2f)', coords.x, coords.y, coords.z))
    print(string.format('Heading: %.2f', heading))
    print(string.format('vector4(%.2f, %.2f, %.2f, %.2f)', coords.x, coords.y, coords.z, heading))
    print('==========================')

    ShowNotification(string.format('~g~Coords copied to console~n~~w~X: %.2f Y: %.2f Z: %.2f', coords.x, coords.y, coords.z))
end)
